/*
 * $Id: 6581_.cxx,v 1.8 1996/04/02 14:05:34 ms Exp $
 * 
 * 
 * MOS-6581 Emulator
 *
 * Copyright (c) 1995 Michael Schwendt
 * All rights reserved.
 * InterNet email: 3schwend@informatik.uni-hamburg.de
 *
 * Redistribution and use in source and binary forms, either unchanged or
 * modified, are permitted provided that the following conditions are met: 
 * 
 * (1) Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * (2) Redistributions in binary form must reproduce the above copyright 
 * notice, this list of conditions and the following disclaimer in the 
 * documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED 
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  known bugs (or missing features, compared to the DOS assembly version):
 *   - sloppy port of Galway-Noise
 *
 * $Log: 6581_.cxx,v $
 * Revision 1.8  1996/04/02 14:05:34  ms
 * Changed/improved the emulation of ring modulation
 *
 * Revision 1.7  1996/03/27 01:08:26  ms
 * TEST: Added ability to fake a plain C64 memory for PlaySID compatibility
 *
 * Revision 1.6  1996/03/25 05:09:06  ms
 * *** empty log message ***
 *
 * Revision 1.5  1996/03/06 15:20:20  ms
 * Fixed forcentsc speed setting
 *
 * Revision 1.4  1996/02/28 00:36:19  ms
 * Changed the sidemufillbuffer() to be a friend of both,
 * the sidemuconfig and the sidtune class
 * Removed some global variables
 *
 * Revision 1.3  1996/02/21 02:24:26  ms
 * Previous revision 1.2.1.1
 * Changes to 8/16-bit and different sample encodings
 * Fixed sidemufillbuffer()'s problems with small buffers
 *
 * Revision 1.2.1.1  1996/02/17 23:10:19  ms
 * Fixed 16-bit and stereo fillroutines
 *
 * Revision 1.2  1996/02/16 01:05:22  ms
 * Added temporary support for different sample formats
 * Added support for new sidemuconfig class
 *
 * Revision 1.1  1996/02/11 21:31:15  ms
 * Fixed order in sidmode70(); thanks <patrick@usn.nl>
 * AmplModTable no longer external; moved from envelope.cxx
 * Dropped <stdio.h> usage
 * Changed fillbuffer() according to external changes
 * for C64 and PlaySID compatibility
 *
 * Revision 1.0  1996/01/21 04:12:05  ms
 * Initial revision
 *
 */

#include <iostream.h>
#include <iomanip.h>

#include "mytypes.h"
#include "6581.h"
#include "sidtune.h"
#include "m68k.h"
#include "opstruct.h"
#include "regeval.h"
#include "samples.h"
#include "6510.h"

typedef ubyte (*ptr2sidfunc)(struct sidoperator *);

extern void sidfrequenzcalc(struct sidoperator*);

extern "C" void sidsetreplayingspeed(byte, byte);
extern "C" void sidchangereplayingspeed(void);

sidoperator optr1, optr2, optr3;

extern ubyte sidkeys[];
extern ubyte* c64mem1;
extern ubyte* c64mem2;  

extern ubyte MasterVolume; 

byte *AmplModTable; 

ubyte *DreieckTable;
ubyte *SaegezahnTable;
ubyte *RechteckTable;
ubyte *RauschenTable;

ubyte RMbegrenzer[512];

uword PCMfreq;                      
udword PCMsid, PCMnoisesid;

ubyte zero8bit;
uword zero16bit;
void* fill8bitmono(void*, udword); // only need one fill()-prototype here
void* (*psidemufillfunc)(void*, udword) = &fill8bitmono; // default


int irqflag;
udword tofill;
ubyte bufferscale;

udword SAMPLESadd, SAMPLEScomma;
uword SAMPLES;
  

void sidemufillbuffer( const sidemuconfig& thiscfg,
					   const sidtune& thistune, 
					   void* buffer, udword bufferlen )
{
  // ensure a sane status of the whole emulator
  if ( thiscfg.ready && thistune.status )  {
	uword replayPC = thistune.playaddr;
	ubyte ramrom = thistune.playramrom;
  
	ubyte* c64mem2cp = c64mem2;
	if ( thistune.playsidflag )
	  c64mem2 = c64mem1;
	
	if ( replayPC == 0 )  {
	  if (( replayPC = m68k( c64mem1[0x315], c64mem1[0x314] )) == 0xea31 )  
		replayPC = m68k( c64mem1[0xffff], c64mem1[0xfffe] );
	  if ( thistune.playsidflag )
		ramrom = 5;
	  else
		ramrom = 7;
	}
	
	irqflag = FALSE;
	
	int retcode;
	
	// both, 16-bit and stereo samples take more memory
	// hence fewer samples fit into the buffer
	bufferlen >>= bufferscale;
	while ( bufferlen > 0 )
	{
	  if ( tofill > bufferlen )  
	  {
		buffer = (*psidemufillfunc)(buffer, bufferlen);
		tofill -= bufferlen;
		bufferlen = 0;
	  }
	  else if ( tofill > 0 )  
	  {
		buffer = (*psidemufillfunc)(buffer, tofill);
		bufferlen -= tofill;
		tofill = 0;   
	  }
	  
	  if ( tofill == 0 )
	  {
		if (( thistune.songspeed != 0 ) || thistune.forcentscflag ) 
		  sidchangereplayingspeed();
		
		if ( thistune.playsidflag )  {
		  retcode = interpreter(replayPC, ramrom, 0, 0, 0, TRUE);
		}
		else
		  retcode = interpreter(replayPC, ramrom, 0, 0, 0, FALSE);
		irqflag = TRUE;
		
		register udword temp = ( SAMPLESadd += SAMPLEScomma );
		SAMPLESadd = temp & 0xFFFF;
		tofill = SAMPLES + ( temp > 65535 );
	  }
	} // while bufferlen
	  
	if ( thistune.playsidflag )
	  c64mem2 = c64mem2cp;
	
  } // if status
}



ubyte wavecalcnormal(struct sidoperator*);
ubyte wavecalcringmod(struct sidoperator*);


void sidmodetest(struct sidoperator*);
void sidmode00(struct sidoperator*);
void sidmode10(struct sidoperator*);
void sidmode20(struct sidoperator*);
void sidmode30(struct sidoperator*);
void sidmode40(struct sidoperator*);
void sidmode50(struct sidoperator*);
void sidmode60(struct sidoperator*);
void sidmode70(struct sidoperator*);
void sidmode80(struct sidoperator*);
void sidmode90(struct sidoperator*);
void sidmodeA0(struct sidoperator*);
void sidmodeB0(struct sidoperator*);
void sidmodeC0(struct sidoperator*);
void sidmodeD0(struct sidoperator*);
void sidmodeE0(struct sidoperator*);
void sidmodeF0(struct sidoperator*);


typedef void (*ptr2sidvoidfunc)(struct sidoperator *);


ptr2sidvoidfunc OpModeTable[] =
{
  sidmode00, sidmodetest,
  sidmode10, sidmodetest,
  sidmode20, sidmodetest,
  sidmode30, sidmodetest,
  sidmode40, sidmodetest,
  sidmode50, sidmodetest,
  sidmode60, sidmodetest,
  sidmode70, sidmodetest,
  sidmode80, sidmodetest,
  sidmode90, sidmodetest,
  sidmodeA0, sidmodetest,
  sidmodeB0, sidmodetest,
  sidmodeC0, sidmodetest,
  sidmodeD0, sidmodetest,
  sidmodeE0, sidmodetest,
  sidmodeF0, sidmodetest
};


inline void sidwaveadvance(struct sidoperator* pvoice)
{
  register udword temp = (udword)pvoice->WAVEADDPNT;
  temp += (udword)pvoice->WAVEPNT;
  pvoice->WAVEADDPNT = temp & 0xFFFF;
  pvoice->WAVEADDSTEP += pvoice->WAVESTEP + ( temp > 65535 );
  pvoice->WAVEADDSTEP &= 4095;
}


inline void sidnoiseadvance(struct sidoperator* pvoice)
{
  register udword temp = (udword)pvoice->NOISEADDPNT;
  temp += (udword)pvoice->NOISEPNT;
  pvoice->NOISEADDPNT = temp & 0xFFFF;
  pvoice->NOISEADDSTEP += pvoice->NOISESTEP + ( temp > 65535 );
}


ubyte wavecalcnormal(struct sidoperator* pvoice)
{
  if ( pvoice->FREQCNT == 0 )
  {
    register udword temp = (udword)pvoice -> FREQADDPNT;
    temp += (udword)pvoice->FREQPNT;
    pvoice->FREQADDPNT = temp & 0xFFFF;
    pvoice->FREQCNT = pvoice->FREQLEN + ( temp > 65535 );
    if ( pvoice->FREQCNT == 0 )
    {
      sidmodetest(pvoice);
      return( pvoice->OUTPUT );
    }

    pvoice->WAVESTEP = 4095 / pvoice->FREQCNT;   
    pvoice->WAVEPNT = (( 4095 % pvoice->FREQCNT ) * 65536 )
                        / pvoice->FREQCNT;         

    struct sidoperator* pvoice2 = pvoice->CARRIER;
    if (( pvoice2->SIDCTRL & 2 ) != 0 )
    {
      pvoice2->FREQCNT = 0;    
      pvoice2->WAVEADDSTEP = 0;
      pvoice2->MODCNT = 0;
      pvoice2->MODADDSTEP = 0;
    }
  }
  pvoice->FREQCNT--;
  (*OpModeTable[pvoice->SIDWAVE])(pvoice);
  return(pvoice->OUTPUT);
}



#define dreieck DreieckTable[pvoice->WAVEADDSTEP]
#define saegezahn SaegezahnTable[pvoice->WAVEADDSTEP]
#define rechteck RechteckTable[pvoice->WAVEADDSTEP + pvoice->RECINDEX]
#define rauschen RauschenTable[pvoice->NOISEADDSTEP & 16383]


void sidmodetest(struct sidoperator* pvoice)
{
  pvoice->FREQCNT = 0;
  pvoice->OUTPUT = 0;
}

void sidmode00(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = 0;
}

void sidmode10(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = dreieck;
  sidwaveadvance(pvoice);
}

void sidmode20(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = saegezahn;
  sidwaveadvance(pvoice);
}

void sidmode30(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( dreieck & saegezahn );
  sidwaveadvance(pvoice);
}

void sidmode40(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = rechteck;
  sidwaveadvance(pvoice);
}

void sidmode50(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( dreieck & rechteck );
  sidwaveadvance(pvoice);
}

void sidmode60(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( saegezahn & rechteck );
  sidwaveadvance(pvoice);
}

void sidmode70(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( dreieck & saegezahn & rechteck );
  sidwaveadvance(pvoice);
}

void sidmode80(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = rauschen;
  sidnoiseadvance(pvoice);
}

void sidmode90(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( dreieck & rauschen );
  sidwaveadvance(pvoice);
  sidnoiseadvance(pvoice);
}

void sidmodeA0(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( saegezahn & rauschen );
  sidwaveadvance(pvoice);
  sidnoiseadvance(pvoice);
}

void sidmodeB0(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( dreieck & saegezahn & rauschen );
  sidwaveadvance(pvoice);
  sidnoiseadvance(pvoice);
}

void sidmodeC0(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( rechteck & rauschen );
  sidwaveadvance(pvoice);
  sidnoiseadvance(pvoice);
}

void sidmodeD0(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( dreieck & rechteck & rauschen );
  sidwaveadvance(pvoice);
  sidnoiseadvance(pvoice);
}

void sidmodeE0(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( saegezahn & rechteck & rauschen );
  sidwaveadvance(pvoice);
  sidnoiseadvance(pvoice);
}

void sidmodeF0(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = ( dreieck & saegezahn & rechteck & rauschen );
  sidwaveadvance(pvoice);
  sidnoiseadvance(pvoice);
}




void RMsidmodetest(struct sidoperator*);
void RMsidmode00(struct sidoperator*);
void RMsidmode10(struct sidoperator*);
void RMsidmode30(struct sidoperator*);
void RMsidmode50(struct sidoperator*);
void RMsidmode70(struct sidoperator*);


ptr2sidvoidfunc RMOpModeTable[] =
{
  RMsidmode00, RMsidmodetest,
  RMsidmode10, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode30, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode50, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode70, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode00, RMsidmodetest,
  RMsidmode00, RMsidmodetest
};


inline void sidringadvance(struct sidoperator* pvoice)
{
  register udword temp = ( pvoice->MODADDPNT += pvoice->MODPNT );
  pvoice->MODADDPNT = temp & 0xFFFF;
  pvoice->MODADDSTEP += pvoice->MODSTEP + ( temp > 65535 );
  pvoice->MODADDSTEP = pvoice->MODADDSTEP & 4095;
}


ubyte wavecalcringmod(struct sidoperator* pvoice)
{
  if ( pvoice->FREQCNT == 0 )  {
    register udword temp = (udword)pvoice -> FREQADDPNT;
    temp += (udword)pvoice->FREQPNT;
    pvoice->FREQADDPNT = temp & 0xFFFF;
    pvoice->FREQCNT = pvoice->FREQLEN + ( temp > 65535 );
    if ( pvoice->FREQCNT == 0 )  {
      sidmodetest(pvoice);
      return( pvoice->OUTPUT );
    }

    pvoice->WAVESTEP = 4095 / pvoice->FREQCNT;   
    pvoice->WAVEPNT = (( 4095 % pvoice->FREQCNT ) * 65536 )
                        / pvoice->FREQCNT;         

    struct sidoperator* pvoice2 = pvoice->CARRIER;
    if (( pvoice2->SIDCTRL & 2 ) != 0 )  {
      pvoice2->FREQCNT = 0;    
      pvoice2->WAVEADDSTEP = 0;
      pvoice2->MODCNT = 0;
      pvoice2->MODADDSTEP = 0;
    }
  }
  pvoice->FREQCNT--;
  
  if ( pvoice->MODCNT == 0 )  {
    register udword temp = pvoice->MODADDPNT + pvoice->MODPNT;
    pvoice->MODADDPNT = temp & 0xFFFF;
    pvoice->MODCNT = pvoice->MODFREQLEN + ( temp > 65535 );
    if ( pvoice->MODCNT == 0 )  {
       pvoice->MODPNT = 0;
       pvoice->MODADDPNT = 0;
    }
    else  {
      pvoice->MODSTEP = 4095 / pvoice->MODCNT;   
      pvoice->MODPNT = (( 4095 % pvoice->MODCNT ) * 65536 )
                          / pvoice->MODCNT;     

      struct sidoperator* pvoice2 = pvoice->CARRIER;
      if (( pvoice2->SIDCTRL & 2 ) != 0 )  {
        pvoice2->FREQCNT = 0; 
        pvoice2->WAVEADDSTEP = 0;
        pvoice2->MODCNT = 0;
        pvoice2->MODADDSTEP = 0;
      }
    }
  }
  pvoice->MODCNT--;

  (*RMOpModeTable[pvoice->SIDWAVE])(pvoice);
  sidwaveadvance(pvoice);
  sidringadvance(pvoice);
  return( pvoice->OUTPUT );
}


#define rm_dreieck DreieckTable[(pvoice->WAVEADDSTEP + 2048) & 4095]
#define rm_saegezahn SaegezahnTable[(pvoice->WAVEADDSTEP + 2048) & 4095]
#define rm_rechteck RechteckTable[(pvoice->WAVEADDSTEP + 4096) & 4095]

void RMsidmodetest(struct sidoperator* pvoice)
{
  pvoice->FREQCNT = 0;
  pvoice->MODCNT = 0;
  pvoice->OUTPUT = 0;
}

void RMsidmode00(struct sidoperator* pvoice)
{
  pvoice->OUTPUT = 0;
}

void RMsidmode10(struct sidoperator* pvoice)
{
  if ( RechteckTable[pvoice->MODADDSTEP + 2048] == 127 )
	pvoice->OUTPUT = dreieck;
  else  {
	pvoice->OUTPUT = rm_dreieck;
  }
}

void RMsidmode30(struct sidoperator* pvoice)
{
  if ( RechteckTable[pvoice->MODADDSTEP + 2048] == 127 )
	pvoice->OUTPUT = dreieck & saegezahn;
  else
	pvoice->OUTPUT = rm_dreieck & rm_saegezahn;
}

void RMsidmode50(struct sidoperator* pvoice)
{
  if ( RechteckTable[pvoice->MODADDSTEP + 2048] == 127 )
	pvoice->OUTPUT = dreieck & rechteck;
  else
	pvoice->OUTPUT = rm_dreieck & rm_rechteck;
}

void RMsidmode70(struct sidoperator* pvoice)
{
  if ( RechteckTable[pvoice->MODADDSTEP + 2048] == 127 )
	pvoice->OUTPUT = dreieck & saegezahn & rechteck;
  else
	pvoice->OUTPUT = rm_dreieck & rm_saegezahn & rm_rechteck;
}




inline void sidemupreparevoices()
{
  MasterVolume = ( c64mem2[0xd418] & 15 );

  optr1.ADSRCTRL |= ( sidkeys[4] & 1 );         
  sidemuset( &optr1, 0xd400 );
  optr2.ADSRCTRL |= ( sidkeys[11] & 1 );        
  sidemuset( &optr2, 0xd407 );
  optr3.ADSRCTRL |= ( sidkeys[18] & 1 );        
  sidemuset( &optr3, 0xd40e );

  sidfrequenzcalc( &optr1 );
  sidfrequenzcalc( &optr2 );
  sidfrequenzcalc( &optr3 );
}


void* fill8bitmono( void* buffer, udword bufferlen )
{
  sidemupreparevoices();
  SID_CheckForSampleEmuInit();
  ubyte* buffer8bit = (ubyte*)buffer;
  
  for ( ; bufferlen > 0; bufferlen-- )
  {
    *buffer8bit = zero8bit + AmplModTable[ 256 * (*optr1.ADSRSUB)(&optr1)
                                           + (*optr1.SIDWAVEROUT)(&optr1) ];
    *buffer8bit += AmplModTable[ 256 * (*optr2.ADSRSUB)(&optr2)
                                 + (*optr2.SIDWAVEROUT)(&optr2) ];
    *buffer8bit += AmplModTable[ 256 * (*optr3.ADSRSUB)(&optr3)
                                 + (*optr3.SIDWAVEROUT)(&optr3) ];
    *buffer8bit += AmplModTable[ 0xFF00 + (*SampleEmuRout)() ];
    buffer8bit++;
  }
  return(buffer8bit);
}

void* fill8bitstereo( void* buffer, udword bufferlen )
{
  sidemupreparevoices();
  SID_CheckForSampleEmuInit();
  ubyte* buffer8bit = (ubyte*)buffer;
  
  for ( ; bufferlen > 0; bufferlen-- )
  {
    *buffer8bit = zero8bit + AmplModTable[ 256 * (*optr1.ADSRSUB)(&optr1)
                                           + (*optr1.SIDWAVEROUT)(&optr1) ];
    *buffer8bit += AmplModTable[ 256 * (*optr3.ADSRSUB)(&optr3)
                                 + (*optr3.SIDWAVEROUT)(&optr3) ];
    buffer8bit++;
    *buffer8bit = zero8bit + AmplModTable[ 256 * (*optr2.ADSRSUB)(&optr2)
                                           + (*optr2.SIDWAVEROUT)(&optr2) ];
    *buffer8bit += AmplModTable[ 0xFF00 + (*SampleEmuRout)() ];
    buffer8bit++;
  }
  return(buffer8bit);
}


void* fill16bitmono( void* buffer, udword bufferlen )
{
  sidemupreparevoices();
  SID_CheckForSampleEmuInit();
  word* buffer16bit = (word*)buffer;

  for ( ; bufferlen > 0; bufferlen-- )
  {
    *buffer16bit = zero16bit + 256 * AmplModTable[ 256 * (*optr1.ADSRSUB)(&optr1)
                                                   + (*optr1.SIDWAVEROUT)(&optr1) ];
    *buffer16bit += 256 * AmplModTable[ 256 * (*optr2.ADSRSUB)(&optr2)
                                        + (*optr2.SIDWAVEROUT)(&optr2) ];
    *buffer16bit += 256 * AmplModTable[ 256 * (*optr3.ADSRSUB)(&optr3)
                                        + (*optr3.SIDWAVEROUT)(&optr3) ];
    *buffer16bit += 256 * AmplModTable[ 0xFF00 + (*SampleEmuRout)() ];
    buffer16bit++;
  }
  return(buffer16bit);
}

void* fill16bitstereo( void* buffer, udword bufferlen )
{
  sidemupreparevoices();
  SID_CheckForSampleEmuInit();
  word* buffer16bit = (word*)buffer;

  for ( ; bufferlen > 0; bufferlen-- )
  {
    *buffer16bit = zero16bit + 256 * AmplModTable[ 256 * (*optr1.ADSRSUB)(&optr1)
                                                   + (*optr1.SIDWAVEROUT)(&optr1) ];
    *buffer16bit += 256 * AmplModTable[ 256 * (*optr3.ADSRSUB)(&optr3)
                                        + (*optr3.SIDWAVEROUT)(&optr3) ];
    buffer16bit++;
    *buffer16bit = zero16bit + 256 * AmplModTable[ 256 * (*optr2.ADSRSUB)(&optr2)
                                                   + (*optr2.SIDWAVEROUT)(&optr2) ];
    *buffer16bit += 256 * AmplModTable[ 0xFF00 + (*SampleEmuRout)() ];
    buffer16bit++;
  }
  return(buffer16bit);
}
